<?php
namespace wbb\data\post\rating;

use wbb\data\post\Post;
use wcf\data\AbstractDatabaseObjectAction;
use wcf\data\IToggleAction;
use wcf\data\user\User;
use wcf\system\database\util\PreparedStatementConditionBuilder;
use wcf\system\image\ImageHandler;
use wcf\util\FileUtil;
use wcf\system\exception\SystemException;
use wcf\system\exception\PermissionDeniedException;
use wcf\system\exception\IllegalLinkException;
use wcf\system\exception\UserInputException;
use wcf\system\WCF;
use wbb\system\cache\builder\PostRatingCacheBuilder;
use wcf\system\upload\DefaultUploadFileValidationStrategy;

/**
 * Executes post rating-related actions.
 *
 * @author     Cyperghost
 * @copyright  2016 (C) by Cyperghost
 *
 * @package    de.wbb-elite.post.rating
 * @category   WoltLab Burning Board
 */
class PostRatingAction extends AbstractDatabaseObjectAction implements IToggleAction{
	/**
	 * @see \wcf\data\AbstractDatabaseObjectAction::$className
	 */
	protected $className = 'wbb\data\post\rating\PostRatingEditor';

	/**
	 * @see \wcf\data\AbstractDatabaseObjectAction::$permissionsCreate
	 */
	protected $permissionsCreate = array('admin.content.canManagePostRating');

	/**
	 * @see \wcf\data\AbstractDatabaseObjectAction::$permissionsDelete
	 */
	protected $permissionsDelete = array('admin.content.canManagePostRating');

	/**
	 * @see \wcf\data\AbstractDatabaseObjectAction::$permissionsUpdate
	 */
	protected $permissionsUpdate = array('admin.content.canManagePostRating');

	/**
	 * @see \wcf\data\AbstractDatabaseObjectAction::$requireACP
	 */
	protected $requireACP = array('create', 'delete', 'update', 'upload');
	/**
	 * @see    \wcf\data\AbstractDatabaseObjectAction::$allowGuestAccess
	 */
	protected $allowGuestAccess = array('getRatingList');

	/**
	 * post rating object
	 *
	 * @var \wbb\data\post\rating\PostRating
	 */
	public $postRating = null;


	/**
	 * @see \wcf\data\AbstractDatabaseObjectAction::create()
	 */
	public function create(){
		$postRating = parent::create();

		// handle post rating icon preview image
		$tmp = WBB_DIR.'images/post/rating/ratingPreview-'.$this->parameters['tmpHash'].'.'.$this->parameters['extension'];
		$fileName = $this->parameters['fileName'];
		$this->updatePreviewImage($postRating, $fileName, $tmp);

		return $postRating;
	}

	/**
	 * @see \wcf\data\AbstractDatabaseObjectAction::delete()
	 */
	public function delete(){
		foreach($this->objects as $postRating){
			// remove preview image
			$previewImage = WBB_DIR.$postRating->image;
			if(file_exists($previewImage)){
				@unlink($previewImage);
			}
		}

		$count = parent::delete();

		return $count;
	}

	/**
	 * Handles uploaded preview images.
	 *
	 * @return array<string>
	 */
	public function upload(){
		// save files
		$files = $this->parameters['__files']->getFiles();
		$file = $files[0];

		try{
			if(!$file->getValidationErrorType()){
				// shrink avatar if necessary
				$fileLocation = $file->getLocation();
				$imageData = getimagesize($fileLocation);
				if($imageData[0] > PostRating::PREVIEW_IMAGE_MAX_HEIGHT || $imageData[1] > PostRating::PREVIEW_IMAGE_MAX_WIDTH){
					try{
						$adapter = ImageHandler::getInstance()->getAdapter();
						$adapter->loadFile($fileLocation);
						$fileLocation = FileUtil::getTemporaryFilename();
						$thumbnail = $adapter->createThumbnail(PostRating::PREVIEW_IMAGE_MAX_HEIGHT, PostRating::PREVIEW_IMAGE_MAX_WIDTH, false);
						$adapter->writeImage($thumbnail, $fileLocation);
						$imageData = getimagesize($fileLocation);
					}catch(SystemException $e){
						throw new UserInputException('image');
					}
				}

				// check avatar directory
				// and create subdirectory if necessary
				$dir = dirname($file->getLocation());
				if(!@file_exists($dir)){
					FileUtil::makePath($dir, 0777);
				}

				// move uploaded file
				if(copy($fileLocation, WBB_DIR.'images/post/rating/ratingPreview-'.$this->parameters['tmpHash'].'.'.$file->getFileExtension())){
					@unlink($fileLocation);

					if($this->parameters['postRatingID']){
						$this->updatePreviewImage($this->postRating, $file->getFilename(), WBB_DIR.'images/post/rating/ratingPreview-'.$this->parameters['tmpHash'].'.'.$file->getFileExtension());

						return array('extension' => $file->getFileExtension(),
									 'url'       => WCF::getPath('wbb').'images/post/rating/'.$file->getFilename(),
									 'fileName'  => $file->getFilename()
						);
					}

					// return result
					return array('extension' => $file->getFileExtension(),
								 'url'       => WCF::getPath('wbb').'images/post/rating/ratingPreview-'.$this->parameters['tmpHash'].'.'.$file->getFileExtension(),
								 'fileName'  => $file->getFilename()
					);
				}else{
					throw new UserInputException('image', 'uploadFailed');
				}
			}
		}catch(UserInputException $e){
			$file->setValidationErrorType($e->getType());
		}

		return array('errorType' => $file->getValidationErrorType());
	}

	/**
	 * Updates post rating preview image.
	 *
	 * @param \wbb\data\post\rating\PostRating $postRating
	 * @param string                           $fileName
	 * @param string                           $tmp
	 */
	protected function updatePreviewImage(PostRating $postRating, $fileName, $tmp){
		if(!isset($this->parameters['tmpHash'])){
			return;
		}
		$oldFilename = WBB_DIR.$postRating->image;
		if(!empty($postRating->image) && file_exists($oldFilename)){
			@unlink(WBB_DIR.'images/post/rating/'.$postRating->image);
			if(@copy($tmp, WBB_DIR.'images/post/rating/'.$fileName)){

				// update filename in database
				$sql = 'UPDATE wbb'.WCF_N.'_post_rating
              SET image = ?
              WHERE ratingID = ?';
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute(array(
					'images/post/rating/'.$fileName,
					$postRating->ratingID
				));

			}else{
				// remove temp file
				@unlink($oldFilename);
			}
		}else{
			if(@copy($tmp, WBB_DIR.'images/post/rating/'.$fileName)){

				// update filename in database
				$sql = 'UPDATE wbb'.WCF_N.'_post_rating
              SET image = ?
              WHERE ratingID = ?';
				$statement = WCF::getDB()->prepareStatement($sql);
				$statement->execute(array(
					'images/post/rating/'.$fileName,
					$postRating->ratingID
				));

			}
		}
	}

	/**
	 * Validates the upload action.
	 */
	public function validateUpload(){
		// check upload permissions
		if(!WCF::getSession()->getPermission('admin.content.canManagePostRating')){
			throw new PermissionDeniedException();
		}

		$this->readString('tmpHash');
		$this->readInteger('ratingID', true);

		if($this->parameters['postRatingID']){
			$postRating = new PostRating($this->parameters['postRatingID']);
			if(!$postRating->ratingID){
				throw new IllegalLinkException();
			}

			$this->postRating = $postRating;
		}

		if(count($this->parameters['__files']->getFiles()) != 1){
			throw new IllegalLinkException();
		}

		// check max filesize, allowed file extensions etc.
		$this->parameters['__files']->validateFiles(new DefaultUploadFileValidationStrategy(PHP_INT_MAX, array('jpg',
																											   'jpeg',
																											   'png')));
	}

	/**
	 * @see \wcf\data\IToggleAction::toggle()
	 */
	public function toggle(){
		foreach($this->objects as $postRating){
			$postRating->update(array(
				'isDisabled' => $postRating->isDisabled ? 0 : 1
			));
		}
	}

	/**
	 * @see \wcf\data\IToggleAction::validateToggle()
	 */
	public function validateToggle(){
		parent::validateUpdate();
	}

	/**
	 * @see \wcf\data\IToggleAction::toggle()
	 */
	public function rating(){
        $postRatings = PostRatingCache::getInstance();
        $data = $postRatings->getPostRatingsByPostID(intval($this->parameters['postID']));
        $ratings_new = '';
 		$post = new \wbb\data\post\Post($this->parameters['postID']);

        if(isset($data["user"]) && isset($data["user"][WCF::getUser()->userID])){
            $sql = 'DELETE FROM wbb'.WCF_N.'_post_rating_to_post WHERE postID = ? AND userID = ?';
            $statement = WCF::getDB()->prepareStatement($sql);
            $statement->execute(array(intval($this->parameters['postID']),
                                      WCF::getUser()->userID));
        }
 		if (!(isset($data["user"]) && isset($data["user"][WCF::getUser()->userID]) && isset($data["user"][WCF::getUser()->userID][$this->objectIDs[0]]))){
 	 		$sql = 'INSERT INTO wbb'.WCF_N.'_post_rating_to_post VALUES (?, ?, ?, ?)';
            $statement = WCF::getDB()->prepareStatement($sql);
            $statement->execute(array($this->objectIDs[0], intval($this->parameters['postID']),
                                      WCF::getUser()->userID, $post->userID));
 		}

        foreach($postRatings->getPostRatings() as $postRating){
        	$sql = 'SELECT COUNT(*) as counter FROM wbb'.WCF_N.'_post_rating_to_post WHERE ratingID = ? and postID = ?';
			$statement = WCF::getDB()->prepareStatement($sql);
			$statement->execute(array($postRating->ratingID, $post->postID));
			$count = $statement->fetchArray()["counter"];

        	if($count > 0){
        		$ratings_new .= '
	                <span class="postRating jsTooltip pointer" title="' . $postRating->getTitle() . '" data-user-id="' . $post->userID . '" data-post-id="' . $post->postID . '" data-object-id="' . $postRating->ratingID . '">
	                    <img src="' . $postRating->getPreviewImage() . '" style="width: 16px; height: 16px" />
	                    <sub>x' . $count . '</sub>
	                </span>
	            ';
        	}
        }

        if(empty($ratings_new)){
            $ratings_new = '&nbsp;';
        }else{
        	//$ratings_new .= '&nbsp;<a class="showPostRating" data-object-id="' . $post->postID . '">(Liste)</a>';
        }

        PostRatingCacheBuilder::getInstance()->reset();

        return array('ratings_new' => $ratings_new, 'post_id' => $post->postID);
    }

	/**
	 * Validates rating.
	 */
	public function validateRating(){
		if(!WCF::getUser()->userID){
			throw new PermissionDeniedException();
		}
		$this->readInteger("postID");
		$post = new Post(intval($this->parameters['postID']));
		if(!$post->canRead()){
			throw new PermissionDeniedException();
		}
		if($post->userID == WCF::getUser()->userID){
			throw new PermissionDeniedException();
		}
	}

	/**
	 * Validates get rating list
	 */
	public function validateGetRatingList(){
		$this->readInteger("postID");
		$post = new Post(intval($this->parameters['postID']));
		if(!$post->canRead()){
			throw new PermissionDeniedException();
		}
	}

	/**
	 * Return the template for rating popup list
	 *
	 * @return array<mixed>
	 */
	public function getRatingList(){
		$cache = PostRatingCache::getInstance();

		$list = array();

		// get all post ratinges
		$sql = 'SELECT rating_to_post.ratingID as ratingID, rating_to_post.userID as userID FROM wbb'.WCF_N.'_post_rating_to_post rating_to_post
		LEFT JOIN wbb'.WCF_N.'_post_rating rating ON(rating.ratingID = rating_to_post.ratingID)
		WHERE rating_to_post.postID = ?';
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute(array(intval($this->parameters['postID'])));
		while($row = $statement->fetchArray()){
			$list[$row["ratingID"]][] = new User($row["userID"]);
		}

		WCF::getTPL()->assign(array('RatingList' => $list, 'postRating' => $cache->getPostRatings()));


		return array('template' => WCF::getTPL()->fetch('postRatingList', 'wbb'), 'title' => WCF::getLanguage()->get("wbb.post.rating.list.title"));
	}
}