<?php
namespace wcf\util;

use wcf\data\user\UserProfileList;
use wcf\system\cache\builder\UserGroupCacheBuilder;
use wcf\system\language\LanguageFactory;
use wcf\system\WCF;

/**
 * Group mention util class
 *
 * @author        Olaf Braun
 * @copyright     2016 Olaf Braun - Software Development
 *
 * @package       wcf\util
 */
class GroupMentionUtil{
	/**
	 * Return a list of mention group´s
	 *
	 * @param $text string
	 *
	 * @return    array<string>
	 */
	public static function getMentionedGroups($text){
		// remove quotes
		$newText = $text;
		if(preg_match_all("~(?:\[quote|\[/quote\])~i", $text, $matches)){
			$newText = '';
			$substrings = preg_split("~(?:\[quote|\[/quote\])~i", $text);

			$inQuote = 0;
			foreach($matches[0] as $i => $tag){
				if(!$inQuote){
					$newText .= $substrings[$i];
				}

				if($tag == '[quote'){
					$inQuote++;
				}else{
					$inQuote--;
				}
			}

			if(!$inQuote) $newText .= $substrings[count($substrings) - 1];
		}

		// check for group mentions
		$groupsFound = array();
		$regex = self::buildRegex();
		$nameToGroup = $regex[0];
		if(preg_match_all("~@(".$regex[1].")~", $newText, $matches)){
			foreach($matches[1] as $groupName){
				if(isset($nameToGroup[$groupName])){
					$groupsFound[] = $nameToGroup[$groupName];
				}
			}
		}
		return $groupsFound;
	}

	/**
	 * Find users to send the notification to them
	 *
	 * @param array $groups
	 * @param integer  $userID
	 *
	 * @return array
	 */
	public static function findUsers(array $groups, $userID = null){
		// get user´s profiles
		$userList = new UserProfileList();
		$userList->sqlJoins .= " LEFT JOIN wcf".WCF_N."_user_to_group user_to_group ON(user_table.userID = user_to_group.userID)";
		$userList->sqlConditionJoins .= " LEFT JOIN wcf".WCF_N."_user_to_group user_to_group ON(user_table.userID = user_to_group.userID)";
		$userList->getConditionBuilder()->add('user_to_group.groupID IN (?)', array($groups));
		if($userID != null){
			$userList->getConditionBuilder()->add('user_table.userID <> ?', array($userID));
		}
		$userList->readObjects();
		return $userList->getObjects();
	}

	/**
	 * Return the list of new group mentions
	 *
	 * @param $oldText string
	 * @param $newText string
	 *
	 * @return array
	 */
	public static function getNewMention($oldText, $newText){
		$oldMatches = $newMatches = array();
		$regex = self::buildRegex();
		$nameToGroup = $regex[0];
		if(preg_match_all("~@(".$regex[1].")~", $oldText, $matches)){
			foreach($matches[1] as $groupName){
				if(isset($nameToGroup[$groupName])){
					$oldMatches[] = $nameToGroup[$groupName];
				}
			}
		}
		if(preg_match_all("~@(".$regex[1].")~", $newText, $matches)){
			foreach($matches[1] as $groupName){
				if(isset($nameToGroup[$groupName])){
					$newMatches[] = $nameToGroup[$groupName];
				}
			}
		}
		return array_diff($newMatches, $oldMatches);
	}

	/**
	 * Return the regex string to find group names
	 *
	 * @return array
	 * @throws \wcf\system\exception\SystemException
	 */
	public static function buildRegex(){
		$groups = UserGroupCacheBuilder::getInstance()->getData(array(), 'groups');
		$languages = LanguageFactory::getInstance()->getLanguages();
		$names = array();
		$nameToGroup = array();
		foreach($groups as $group){
			if(in_array($group->groupID, array(explode(",", DE_HOSTINGTIME_GROUP_MENTION_GROUPS)))) continue;
			if(count($languages) > 1){
				$groupNames = self::getLanguage($group->groupName);
				foreach($groupNames as $name){
					$nameToGroup[$name] = $group->groupID;
				}
				$names = array_merge($names, $groupNames);
			}else{
				$nameToGroup[$group->groupName] = $group->groupID;
				$names[] = $group->groupName;
			}
		}
		return array($nameToGroup, implode("|", $names));
	}

	/**
	 * Return the list of group names
	 *
	 * @param $name string
	 *
	 * @return array
	 * @throws \wcf\system\database\DatabaseException
	 */
	private static function getLanguage($name){
		$result = array();
		$sql = "SELECT languageItemValue FROM wcf".WCF_N."_language_item WHERE languageItem = ?";
		$statement = WCF::getDB()->prepareStatement($sql);
		$statement->execute(array($name));
		while($row = $statement->fetchArray()){
			$result[] = $row["languageItemValue"];
		}
		if(count($result) == 0){
			return array($name);
		}

		return $result;
	}
}